function eng = importEngine(dataFile, nvp)
arguments
    dataFile string = [];
    nvp.method {mustBeMember(nvp.method, ["gridfit", "scatteredInterpolant"])} = "gridfit";
    nvp.plots = true;
end

if isempty(dataFile)
    [dataFile, path] = uigetfile({'*.xlsx;*.xls',...
        'Excel Spreadsheet (*.xlsx,*.xls)';
        '*.*',  'All Files (*.*)'}, ...
        'Select a File');
    dataFile = fullfile(path, dataFile);
end
[~, dataFileName] = fileparts(dataFile);

%% Read main parameters from the "_main" sheet in the engine data spreadsheet

opts = detectImportOptions(dataFile,'Sheet','main', 'ReadRowNames',1);
rawdata = readtable(dataFile, opts);

eng.shortName = dataFileName;
eng.displacement = rawdata.value('Displacement')*1e-3; % m^3
eng.ratedPwr = rawdata.value('Rated power'); % kW
eng.inertia = rawdata.value('Moment of inertia'); % kg*m^2
eng.idleSpd = rawdata.value('Idle speed') * pi/30; % rad/s
eng.maxSpd = rawdata.value('Max speed') * pi/30; % rad/s
eng.mass = rawdata.value('Mass'); % kg
eng.fuelLHV = rawdata.value('Fuel lower heating value'); % J/kg
eng.fuelDensity = rawdata.value('Fuel density'); % kg/l
eng.CO2ttwFactor = rawdata.value('CO2ttw factor'); % kgCO2/kgFuel

% Read full name
tmp = readcell(dataFile,'Sheet','main');
[i,j] = find(strcmp(tmp,'Full Name'));
eng.fullName = tmp(i,j+1);

% Read max and motoring torque characteristics
opts = detectImportOptions(dataFile,'Sheet','lim');
opts.VariableUnitsRange = 'A2:C2';
limdata = readtable(dataFile, opts);

% Store maxTorque and motTorque interpolants, Nm = f(rad/s)
eng.maxTrq = griddedInterpolant(limdata.speed.*pi/30, limdata.maxTorque);
eng.motTrq = griddedInterpolant(limdata.speed.*pi/30, limdata.motoringTorque);

%% Generate mapped data (fuel consumption and emissions)
% Read data for fc map generation from the "map" sheet in the engine data spreadsheet
opts = detectImportOptions(dataFile,'Sheet','map');
opts.VariableUnitsRange = "A2";
mapData = readtable(dataFile, opts);
eng.mapData = mapData;

% Interpolate data over a newly defined grid

% Create new uniform grid
spdBrk = linspace(eng.idleSpd, eng.maxSpd, 100); % rad/s
trqBrk = linspace(min(limdata.motoringTorque), max(limdata.maxTorque), 100);

species = ["fuelFlwRate", "NOx", "HC", "CO", "PM", "exhFlwRate", "exhTemp"];

for n = 1:length(species)
    eng.(species(n)) = createMap(species(n), mapData, spdBrk, trqBrk, nvp.method, nvp.plots, eng);
end

species_spec = ["bsfc", "bsNOx", "bsHC", "bsCO", "bsPM"];

for n = 1:length(species_spec)
    eng.(species_spec(n)) = createSpecMap(species(n), eng);
end


end

function map = createMap(specie, mapData, spdBrk, trqBrk, method, plots, eng)

mapData.speed = mapData.speed .* pi/30; % rad/s

[~, col_idx] = ismember(specie, mapData.Properties.VariableNames);
if col_idx > 0
    % Convert flow rates to kg/s and temperature to K
    switch mapData.Properties.VariableUnits{col_idx}
        case {'kg/s', 'K'}
            rawData = mapData.(specie);  
        case {'g/s'}
            rawData = mapData.(specie) * 1e-3;
        case {'kg/h'}
            rawData = mapData.(specie) / 3.6e3;
        case {'g/h'}
            rawData = mapData.(specie) / 3.6e6;
        case {'g/kWh'}
            rawData = g_kWh_to_kg_s(mapData.(specie), mapData.speed, mapData.torque);
        case {'C'}
            rawData = mapData.(specie) + 273.15;  
        otherwise
            error("Unrecognized unit " + mapData.Properties.VariableUnits{col_idx} + " for column " + specie)
    end

    % create map
    switch method
        case "scatteredInterpolant"
            [spdMesh, trqMesh] = ndgrid(spdBrk, trqBrk);

            scatterInterp = scatteredInterpolant(mapData.speed, mapData.torque, rawData, 'linear', 'linear'); % kg/s or K
            griddedData = scatterInterp(spdMesh, trqMesh);
            map = griddedInterpolant(spdMesh, trqMesh, griddedData);

        case "gridfit"
            switch specie
                case {"fuelFlwRate", "NOx", "HC", "PM", "exhFlwRate", "exhTemp"}
                    smoothnessFactor = 10;
                    regularizer = 'gradient';
                case "CO"
                    smoothnessFactor = 20;
                    regularizer = 'diffusion';
            end

            [griddedData, spdMesh, trqMesh] = gridfit(mapData.speed, mapData.torque, rawData, ...
                spdBrk, trqBrk, 'extend', 'always', 'smoothness', smoothnessFactor, ...
                'regularizer', regularizer, 'interp', 'triangle');
            spdMesh = spdMesh';
            trqMesh = trqMesh';
            griddedData = griddedData';

            map = griddedInterpolant(spdMesh, trqMesh, griddedData);
    end

else
    map = [];
    plots = false;
end

if plots
    figure
    hold on

    maxTrq = eng.maxTrq(spdMesh);
    griddedData(trqMesh > maxTrq) = nan;

    wotSpd = eng.maxTrq.GridVectors{1};
    wotTrq = eng.maxTrq.Values;
    wotEmis = map(wotSpd, wotTrq);

    if ~strcmp(specie, "exhTemp")
        griddedData = griddedData  * 1e3;
        rawData = rawData * 1e3;
        wotEmis = wotEmis * 1e3;
    else
        griddedData = griddedData - 273.15;
        rawData = rawData - 273.15;
        wotEmis = wotEmis - 273.15;
    end

    surf(spdMesh * 30/pi, trqMesh, griddedData)
    scatter3(mapData.speed * 30/pi, mapData.torque, rawData, 'rx')
    plot3(wotSpd * 30/pi, wotTrq, wotEmis, 'k-', 'LineWidth', 1)
    xlabel("speed, rpm")
    ylabel("torque, Nm")

    if ~strcmp(specie, "exhTemp")
        title(specie + ", g/s")
        legend("Map, g/s", "Data points")
    else
        title(specie + ", °C")
        legend("Map, °C", "Data points")
    end

    figure
    hold on
    contour(spdMesh * 30/pi, trqMesh, griddedData, 'ShowText', 'on')
    scatter(mapData.speed * 30/pi, mapData.torque, 'rx')
    plot(wotSpd * 30/pi, wotTrq, 'k-', 'LineWidth', 1)
    xlabel("speed, rpm")
    ylabel("torque, Nm")
    if ~strcmp(specie, "exhTemp")
        title(specie + ", g/s")
        legend("Map, g/s", "Data points")
    else
        title(specie + ", °C")
        legend("Map, °C", "Data points")
    end
end

end

function map = createSpecMap(specie, eng)

if ~isempty(eng.(specie))
    spdBrk = eng.(specie).GridVectors{1};
    trqBrk = eng.(specie).GridVectors{2};
    val = eng.(specie).Values; % g/s

    [spdMesh, trqMesh] = ndgrid(spdBrk, trqBrk);

    val = kg_s_to_g_kWh(val, spdMesh, trqMesh);

    map = griddedInterpolant(spdMesh, trqMesh, val);
else
    map = [];
end

end

function val = g_kWh_to_kg_s(val, spd, trq)
    % Convert g/kWh to kg/s. Assumes spd in rad/s, trq in Nm.

    pwr = spd .* trq;
    val = val .* pwr ./ 3.6e9;
end

function val = kg_s_to_g_kWh(val, spd, trq)
    % Convert kg/s to g/kWh. Assumes spd in rad/s, trq in Nm.

    pwr = spd .* trq;
    val = val ./ pwr .* 3.6e9;
end